/**  @file bta_find_chessboard.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_find_chessboard.h"
#include <bta_opencv_helper.h>
#include <bta_helper.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifndef BTA_EXCLUDE_FILTERS

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/calib3d/calib3d.hpp>

static std::vector<cv::Point2f> scaleVector(std::vector<cv::Point2f> pointBuf, float resize);


BTA_Status BFLTfindChessboardInit(BTA_FltFindChessboardConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    BTA_FltFindChessboardInst *inst = (BTA_FltFindChessboardInst *)calloc(1, sizeof(BTA_FltFindChessboardInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->channelToProcess = config->channelToProcess;
    inst->scaleFactor = config->scaleFactor;
    inst->edgeCountHor = config->edgeCountHor;
    inst->edgeCountVert = config->edgeCountVert;
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTfindChessboardClose(BTA_FltHandle *handle) {
    BTA_FltFindChessboardInst **inst = (BTA_FltFindChessboardInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTfindChessboardApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltFindChessboardInst *inst = (BTA_FltFindChessboardInst *)handle;
    int chInd;
    int chCount = (*frame)->channelsLen;
    for (chInd = 0; chInd < chCount; chInd++) {
        BTA_Channel *channel = ((*frame)->channels)[chInd];
        if (channel->id == inst->channelToProcess) {
            uint8_t freeChannel = 0;
            if (channel->dataFormat == BTA_DataFormatRgb24 || channel->dataFormat == BTA_DataFormatRgb565) {
                BTA_Channel *channelTemp;
                BTA_Status status = BTAcolorToBw(channel, &channelTemp);
                if (status != BTA_StatusOk) {
                    return status;
                }
                channel = channelTemp;
                freeChannel = 1;
            }
            cv::Mat channelMat = BTAtoMat(channel);
            double minOrig, maxOrig;
            cv::minMaxLoc(channelMat, &minOrig, &maxOrig);
            cv::Mat channelMat8;
            if (std::abs(minOrig) >= 1 && (maxOrig - minOrig != 0)) {
                channelMat.convertTo(channelMat8, CV_8U, 255.0 / (maxOrig - minOrig), -255.0 / minOrig);
            }
            else {
                channelMat.convertTo(channelMat8, CV_8U, 255.0 / maxOrig);
            }
            cv::Mat channelMatUpscaled;
            if (inst->scaleFactor != 1) {
                cv::resize(channelMat8, channelMatUpscaled, cv::Size(0, 0), inst->scaleFactor, inst->scaleFactor);
            }
            else {
                channelMatUpscaled = channelMat8;
            }
            cv::Size s;
            s.width = inst->edgeCountHor;
            s.height = inst->edgeCountVert;
            std::vector<cv::Point2f> pointBuf;
            bool found = findChessboardCorners(channelMatUpscaled, s, pointBuf, CV_CALIB_CB_ADAPTIVE_THRESH | CV_CALIB_CB_FAST_CHECK);
            if (freeChannel) {
                // we created a new intermediate channel before -> free it!
                BTAfreeChannel(&channel);
                channel = ((*frame)->channels)[chInd];
            }
            if (!found) {
                continue;
            }
            cornerSubPix(channelMatUpscaled, pointBuf, cv::Size(5, 5), cv::Size(-1, -1), cv::TermCriteria(CV_TERMCRIT_EPS + CV_TERMCRIT_ITER, 30, 0.1));

            std::vector<cv::Point2f> points = scaleVector(pointBuf, 1.0f / inst->scaleFactor);

            int i;
            float *metadata = (float *)malloc(2 * points.size() * sizeof(float));
            if (metadata) {
                for (i = 0; i < pointBuf.size(); i++) {
                    metadata[2 * i] = points[i].x;
                    metadata[2 * i + 1] = points[i].y;
                }
                BTAinsertMetadataDataIntoChannel(channel, BTA_MetadataIdChessboardCorners, metadata, (uint32_t)(2 * points.size() * sizeof(float)));
            }

            int x, y, k;
            switch (channel->dataFormat) {

            case BTA_DataFormatUInt16:
            case BTA_DataFormatSInt16:
                for (i = 0; i < pointBuf.size(); i++) {
                    for (x = -2; x <= 2; x++) {
                        for (k = -1; k <= 1; k += 2) {
                            y = k * x;
                            //1st iteration: y = -x. 2nd iteration: y = x
                            if (points[i].x + x >= 0 && points[i].x + x < channel->xRes && points[i].y + y >= 0 && points[i].y + y < channel->yRes) {
                                ((uint16_t *)channel->data)[(int)points[i].x + x + ((int)points[i].y + y) * channel->xRes] = 0;
                            }
                        }
                    }
                }
                break;

            case BTA_DataFormatFloat32:
                for (i = 0; i < pointBuf.size(); i++) {
                    for (x = -2; x <= 2; x++) {
                        for (k = -1; k <= 1; k += 2) {
                            y = k * x;
                            //1st iteration: y = -x. 2nd iteration: y = x
                            if (points[i].x + x >= 0 && points[i].x + x < channel->xRes && points[i].y + y >= 0 && points[i].y + y < channel->yRes) {
                                ((uint32_t *)channel->data)[(int)points[i].x + x + ((int)points[i].y + y) * channel->xRes] = 0;
                            }
                        }
                    }
                }
                break;

            case BTA_DataFormatRgb24:
                for (i = 0; i < pointBuf.size(); i++) {
                    for (x = -2; x <= 2; x++) {
                        for (k = -1; k <= 1; k += 2) {
                            y = k * x;
                            //1st iteration: y = -x. 2nd iteration: y = x
                            if (points[i].x + x >= 0 && points[i].x + x < channel->xRes && points[i].y + y >= 0 && points[i].y + y < channel->yRes) {
                                int index = 3 * (int)points[i].x + x + ((int)points[i].y + y) * channel->xRes;
                                ((uint8_t *)channel->data)[index++] = 0x00;
                                ((uint8_t *)channel->data)[index++] = 0xff;
                                ((uint8_t *)channel->data)[index++] = 0x00;
                            }
                        }
                    }
                }
                break;

            case BTA_DataFormatRgb565:
                for (i = 0; i < pointBuf.size(); i++) {
                    for (x = -2; x <= 2; x++) {
                        for (k = -1; k <= 1; k += 2) {
                            y = k * x;
                            //1st iteration: y = -x. 2nd iteration: y = x
                            if (points[i].x + x >= 0 && points[i].x + x < channel->xRes && points[i].y + y >= 0 && points[i].y + y < channel->yRes) {
                                ((uint16_t *)channel->data)[(int)points[i].x + x + ((int)points[i].y + y) * channel->xRes] = 0x07e0;
                            }
                        }
                    }
                    ((uint16_t *)channel->data)[(int)points[i].x + (int)points[i].y * channel->xRes] = 0x07e0;
                }
                break;

            default:
                BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTfindChessboardApply: unsupported data format", channel->dataFormat);
                return BTA_StatusNotSupported;
            }
        }
    }
    return BTA_StatusOk;
}


static std::vector<cv::Point2f> scaleVector(std::vector<cv::Point2f> pointBuf, float resize) {
    std::vector<cv::Point2f> ret(pointBuf.size());
    for (int i = 0; i < pointBuf.size(); i++) {
        ret[i].x = pointBuf[i].x * resize;
        ret[i].y = pointBuf[i].y * resize;
    }
    return ret;
}

#endif
